/**********************************************************************************************************************
 Purpose:
 Creates a test database for performance testing the 4 known good methods for combining DATE and TIME columns.

 The following fully documented programming objects are also created in the database.
    dbo.fnTally          - iTVF  - Integer-based series generator
    dbo.RandInt          - View  - View to allow the creation of random base integers in functions.
    dbo.RandDateTime2    - iTVF  - Returns a column of random DATETIME2's according to parameters.
    dbo.dbo.DateTimeTest - Table - The 10 million row test table (heap)

 This script takes 12-13 seconds to execute on my laptop.
-----------------------------------------------------------------------------------------------------------------------
 Revision History:
 Rev 00 - 28 May 2023 - Jeff Moden
                      - Initial creation and unit test.
 Rev 01 - 03 Jun 2023 - Jeff Moden
                      - Change the database creation code to dynamic SQL
**********************************************************************************************************************/
--===== Identify the section to the operator.
        RAISERROR(N'Starting the run...',0,0) WITH NOWAIT
;

--=====================================================================================================================
--      Code to unconditionally drop the DateTimeTest database for reruns and cleanup.  Commented out for safety.
--=====================================================================================================================
/*
     IF DB_ID('DateTimeTest') IS NOT NULL
  BEGIN
--===== Identify the section to the operator.
        RAISERROR(N'Unconditionally dropping the DateTimeTest database...',0,0) WITH NOWAIT
;
--===== Make sure that we're not currently in the database that we're going to drop.
    USE master
;
--===== Force everyone and everything out of the database so that we can drop it.
  ALTER DATABASE DateTimeTest SET SINGLE_USER WITH ROLLBACK IMMEDIATE;
  ALTER DATABASE DateTimeTest SET MULTI_USER;
   DROP DATABASE DateTimeTest;

    END
;
*/
GO
--=====================================================================================================================
--      CREATE the DateTimeTest database.
--=====================================================================================================================
--===== Identify the section to the operator.
        RAISERROR(N'Creating the DateTimeTest      database...',0,0) WITH NOWAIT
;
--===== Create the database using all default settings.
     -- DROP DATABASE [DateTimeTest];
 CREATE DATABASE [DateTimeTest]
;
--===== Set the database to the SIMPLE recovery model to enable minimal logging.
  ALTER DATABASE DateTimeTest SET RECOVERY SIMPLE WITH NO_WAIT;
GO
--=====================================================================================================================
--      Create the required supporting code in the newly created database.
--=====================================================================================================================
--===== Make the new database the current database.
    USE DateTimeTest
;
--===== Identify the section to the operator.
        RAISERROR(N'Creating the dbo.fnTally       function...',0,0) WITH NOWAIT
;
GO
 CREATE FUNCTION dbo.fnTally
/**********************************************************************************************************************
 Purpose:
 Given a starting value of "0" or "1" and a max positive value <= 2,147,483,647, return the range of whole numbers
 from the start value to the max value.

 This is just one version of the "dbo.fnTally" function. This one is based on 16 original values. It has exactly the
 same functionality as other versions.
-----------------------------------------------------------------------------------------------------------------------
 Usage:
--===== Basic syntax example
 SELECT N FROM dbo.fnTally(@ZeroOrOne,@MaxN); --Note that @ZeroOrOne can only be 0 or 1 as a minor performance boost.

--===== Return values from 0 to 1000
 SELECT N FROM dbo.fnTally(0,1000);

--===== Return values from 1 to 1000
 SELECT N FROM dbo.fnTally(1,1000);
-----------------------------------------------------------------------------------------------------------------------
 1. This code works for SQL Server 2008 and up.
 2. Based on Itzik Ben-Gan's cascading CTE (cCTE) method for creating a "readless" Tally Table source of BIGINTs.
    Refer to the following URL for how it works.
    https://www.itprotoday.com/sql-server/virtual-auxiliary-table-numbers
 3. This code only uses 7 "Nested Loops" in the execution plan instead of 31
-----------------------------------------------------------------------------------------------------------------------
 Revision History:
 Rev 00 - Unknown     - Jeff Moden 
        - Initial creation with error handling for @MaxN.
 Rev 01 - 09 Feb 2013 - Jeff Moden 
        - Modified to start at 0 or 1 and removed error handling for @MaxN.
**********************************************************************************************************************/
        (@ZeroOrOne TINYINT, @MaxN BIGINT)
RETURNS TABLE WITH SCHEMABINDING AS 
 RETURN WITH
   E1(N) AS (SELECT 1 FROM (VALUES (1),(1),(1),(1),(1),(1),(1),(1),(1),(1),(1),(1),(1),(1),(1),(1))E0(N))
             SELECT N = 0 WHERE @ZeroOrOne = 0 UNION ALL --May or may not be present as a row in the return
             SELECT TOP(@MaxN)
                    N = ROW_NUMBER() OVER (ORDER BY (SELECT NULL))
               FROM E1 a,E1 b,E1 c,E1 d,E1 e,E1 f,E1 g,E1 h --0 to 16^8 or 4,294,967,296 rows max
              ORDER BY N
;
GO
--===== Identify the section to the operator.
        RAISERROR(N'Creating the dbo.RandInt       view...',0,0) WITH NOWAIT
;
GO
 CREATE VIEW dbo.RandInt AS
/************************************************************************************************
 Purpose:
 Returns 1 Unconstrained Random Integer for each row the view is CROSS JOINED or APPLY'd to. 
 May be called in functions as a replacment for CHECKSUM(NEWID()).
 
 This code will work in all versions of SQL Server 2008 and up.
=================================================================================================
 Revision History
 Rev 00 - 06 Jun 2004 - Jeff Moden - Initial creation (formally dbo.iFunction)
                      - Initial creation and unit test.
 Rev 01 - 30 Mar 2023 - Jeff Moden 
                      - Rewritten for performance from a view that provided 3 columns.
************************************************************************************************/
 SELECT RandInt = CHECKSUM(NEWID())
; 
GO
--===== Identify the section to the operator.
        RAISERROR(N'Creating the dbo.RandDateTime2 function...',0,0) WITH NOWAIT
;
GO
 CREATE FUNCTION dbo.RandDateTime2
/************************************************************************************************
 Purpose:
 Creates and returns 1 to up to 2 billion (2,147,483,647) random constrained DATETIME2(7) values.
 
 This code will work in all versions of SQL Server 2008 and up.
=================================================================================================
 Example Usage;
--===== Basic syntax
 SELECT RandDateTime2
   FROM dbo.RandDateTime2(@LoDate,@HiDate,@Rows)
;
-------------------------------------------------------------------------------------------------
--===== Create a million row Temp Table on the fly with no indexes with dates 
     -- From '20220701' Inclusive
     --   To '20230701' Exclusive
     -- 587 ms average per Million rows on my laptop. (Tested up to 100 million rows)
   DROP TABLE IF EXISTS #SomeTable;
GO
 SELECT RandDateTime2
   INTO #SomeTable
   FROM dbo.RandDateTime2('20220701','20230701',1000000)
;
-------------------------------------------------------------------------------------------------
--====== "Minimally Logged" INSERT into precreated table with clustered index. 
      -- The RandDateTime2's are NOT guaranteed to be unique.
      -- Requires BULK LOGGED or SIMPLE Recovery Models.
      -- If a FILL FACTOR is assigned, the insert will follow the Fill Factor even if not 100.
-- https://www.sqlservercentral.com/articles/some-t-sql-inserts-do-follow-the-fill-factor-sql-oolie

   DROP TABLE IF EXISTS dbo.SomeTable;
GO
 CREATE TABLE dbo.SomeTable
        (
        RandDateTime2 DATETIME2
        ,INDEX CI_By_RandDateTime2 CLUSTERED (RandDateTime2) WITH (FILLFACTOR=100)
        )
;
 INSERT INTO dbo.SomeTable WITH (TABLOCK) --About 733 ms per million rows at FILLFACTOR = 100.
        (RandDateTime2)
 SELECT RandDateTime2
   FROM dbo.RandDateTime2('20220701','20230701',1000000) --1 MILLION DATETIME2s
  ORDER BY RandDateTime2
 OPTION (RECOMPILE)
;
=================================================================================================
 Depencencies:
 1. dbo.RandInt - View - Necessary because Microsoft doens't allow for the use of NEWID() in a 
                         function because they consider it to be a "side-effecting operator".
=================================================================================================
 Revision History
 Rev 00 - LongTimeAgo - Jeff Moden 
                      - Initial creation and unit test.
 Rev 01 - 30 Mar 2023 - Jeff Moden 
                      - Added documentation for publication
                      - Added usage examples.
                      - Changed Tally function to simplified RowSrc for additional performance.
                      - Changed formulas for additional performance.
                      - Removed WITH SCHEMABINDING to remove dependencies for easy future mods.
                      - Changed to use new view (dbo.RandInt) dedicated to creating Random 
                        Integers using only CHECKSUM(NEWID()) for additional performance and
                        flexibility.
************************************************************************************************/
        (
         @LoDate DATETIME2 --Inclusive
        ,@HiDate DATETIME2 --Exclusive
        ,@Rows   INT       --Limits number of rows to 2,147,483,647 to user prevent accidents.
        )
RETURNS TABLE AS
 RETURN WITH
     E1 AS (SELECT N=1 
              FROM (VALUES (1),(1),(1),(1),(1),(1),(1),(1),(1),(1),(1),(1),(1),(1),(1),(1))E0(N))
,RowSrc AS (SELECT TOP(@Rows) N=NULL --We just need the "Presence of Rows" here. No sequence.
              FROM E1 a,E1 b,E1 c,E1 d,E1 e,E1 f,E1 g,E1 h) --1 to 16^8 or 4,294,967,296 rows
 SELECT RandDateTime2 =
            DATEADD(ns,ABS(ri.RandInt%1000000000) --Nanoseconds per second
            ,DATEADD(ss,ABS(ri.RandInt%86400)     --Seconds per day
             ,DATEADD(dd,ABS(ri.RandInt%DATEDIFF(dd,@LoDate,@HiDate)),@LoDate))) --Day
   FROM RowSrc rs
  CROSS JOIN dbo.RandInt ri
;
GO
/************************************************************************************************
--      Create and populate a test table (HEAP) with random DateTime2 values and the related
--      DATE and TIME columns for various temporal code experiments.
--      Takes about 12 seconds on my laptop. (10 million rows)
--      Jeff Moden - 28 Mar 2023
************************************************************************************************/
--===============================================================================================
--      Presets
--===============================================================================================
--===== Make the current database the DateTimeTest database.
    USE DateTimeTest
;
--===== If the test table already exists, drop it to make reruns easier.
   DROP TABLE IF EXISTS dbo.DateTimeTest;
        CHECKPOINT;
GO
--===== Variables to limit the range of dates used and to control the number of rows created.
     -- TODO: Change the constants in the following DECLARE to suit the needs of your test(s).
DECLARE  @LoDate DATE = '20220701'  -- Inclusive
        ,@HiDate DATE = '20230701'  -- Exclusive
        ,@Rows   INT  = 10000000    -- 10 Million rows
;
--===============================================================================================
--      Create and populate the table (HEAP) on the fly.
--===============================================================================================
--===== Identify the section to the operator.
        RAISERROR(N'Creating the dbo.DateTimeTest  table...',0,0) WITH NOWAIT
;
   WITH cteGenDateTime2 AS
(--==== Generate Random DATETIME2's
 SELECT OriginalDT = RandDateTime2
   FROM dbo.RandDateTime2(@LoDate,@HiDate,@Rows)
)--==== Now that we have a whole date, easily split out the DATE and TIME columns and 
     -- load it all into a table (HEAP).
 SELECT  OriginalDT -- This is the original DATETIME2 for "proofing"
        ,DateValue      = CONVERT(DATE,OriginalDT)
        ,TimeValue      = CONVERT(TIME,OriginalDT)
   INTO dbo.DateTimeTest
   FROM cteGenDateTime2
;
GO
CHECKPOINT;
--===== Inform the operator
        RAISERROR(N'***** Run Complete *****',0,0) WITH NOWAIT;
;
GO
